const PLUGIN_ID = 'signalk-maretron-vmm6';
const PLUGIN_NAME = 'NMEA2000 Maretron VMM6';

module.exports = function (app) {
    const ts_pgns = require("@canboat/ts-pgns"); //<-- Thank you Scott!!! 
    const plugin = {};
    let unsubscribes = [];

    // Switch on/off state enum
    const SwitchState = Object.freeze({
        SwitchOn: 1,
        SwitchOff: 0
    });

    // Verbose Switch on/off state enum
    const verboseOnOff = Object.freeze({
        On: "On",
        Off: "Off"
    });

    // This is needed to store the state of the switch for next key press -
    // since every keypress is an on. The actual On/Off state of the switch -
    // needs to be saved. Not sure how this will work with multiple switches changing the same switch. -
    // might need to wait for callback and parse 127502.
    // Maretron uses two indicators per switch so pathnumber represents that. 
    // numer is the translation for that pathnumber to a real switch
    let switch1 = {state: SwitchState.SwitchOff, number: 1, pathnumer: 1};
    let switch2 = {state: SwitchState.SwitchOff, number: 2, pathnumer: 3};
    let switch3 = {state: SwitchState.SwitchOff, number: 3, pathnumer: 5};
    let switch4 = {state: SwitchState.SwitchOff, number: 4, pathnumer: 7};
    let switch5 = {state: SwitchState.SwitchOff, number: 5, pathnumer: 9};
    let switch6 = {state: SwitchState.SwitchOff, number: 6, pathnumer: 11};
 
    plugin.id = PLUGIN_ID;
    plugin.name = PLUGIN_NAME;
    plugin.description = 'SignalK plugin to allow the use of the Maretron VMM6 with standard NMEA2000 digital switching';
    app.debug('signalk-maretron-vmm6 created')
    plugin.schema = {
        title: PLUGIN_NAME,
        type: 'object',
        properties: {
    	    StatusBank: {
                type: 'number',
                title: 'Status Bank Instance Number:',
                default: 0
    	    },
    	    SwitchBank: {
                type: 'number',
                title: 'Switch Bank Instance Number:',
                default: 0
    	    },
            NumSwitches: {
                type: 'number',
                title: 'Number of switches in bank:',
                default: 0
    	    },
            SwitchType: {
                type: "string",
                title: "Action",
                enum: [
                  "Toggle On/Off",
                  "Toggle On",
                  "Toggle Off",
                  "Momentary"
                ],
                default: "Momentary"
              }
        }
    };

    plugin.start = (options, restartPlugin) => {
        app.debug('Plugin started')
        let localSubscription  = {
        context: 'vessels.self',
        subscribe: [{
            path: `electrical.switches.bank.${options.StatusBank}.${switch1.pathnumer}.state`, 
            period: 25
            },{
            path: `electrical.switches.bank.${options.StatusBank}.${switch2.pathnumer}.state`, 
            period: 25
            },{
            path: `electrical.switches.bank.${options.StatusBank}.${switch3.pathnumer}.state`, 
            period: 25
            },{
            path: `electrical.switches.bank.${options.StatusBank}.${switch4.pathnumer}.state`, 
            period: 25
            },{
            path: `electrical.switches.bank.${options.StatusBank}.${switch5.pathnumer}.state`, 
            period: 25
            },{
            path: `electrical.switches.bank.${options.StatusBank}.${switch6.pathnumer}.state`, 
            period: 25
            }
        ]}

        app.subscriptionmanager.subscribe( localSubscription, unsubscribes, (subscriptionError) => { app.error('Error:' + subscriptionError) }, (delta) => {
            delta.updates.forEach((update) => { 
                if (update.values) {
                    update.values.forEach((value) => {
                        const path = value.path;
                        const key = `${path}.${update.$source}`;                                                 
                        if (path.endsWith('state'))
                        { 
                            //electrical.switches.bank.14.12.state
                            let switchPGN = path.split('.');
                            let instance = Number(switchPGN[3]);
                            let switchNum = Number(switchPGN[4]);                           
                            let sval = value.value;

                            //Needed a common on/off variable for 127502 to send.
                            let switchOnOff = SwitchState.SwitchOff;
                            
                            let switchVal = (sval > 0) ? verboseOnOff.On : verboseOnOff.Off;   

                            //app.debug(`Switch Number: ${switchNum} SwitchVal: ${switchVal} for Path: ${path}`);

                            if (switchVal == verboseOnOff.On){
                                if (instance == options.StatusBank && (switchNum == 1 | switchNum == 3 |switchNum == 5 |switchNum == 7 |switchNum == 9 |switchNum == 11))
                                {
                                    //used below for the 502 send.
                                    let switchNo = 0;
                                    // SwitchVal will always be ON when a key is pressed for momentary switches.
                                    // Need to translate Maretron's use of the 127501 status PGN.                          
                                    switch (switchNum) {
                                        case 1:
                                            switchNo = switch1.number;
                                            switch1.state = (switch1.state === SwitchState.SwitchOff) ? SwitchState.SwitchOn : SwitchState.SwitchOff;  
                                            switchOnOff = switch1.state                                         
                                            break;
                                        case 3:
                                            switchNo = switch2.number;
                                            switch2.state = (switch2.state === SwitchState.SwitchOff) ? SwitchState.SwitchOn : SwitchState.SwitchOff;
                                            switchOnOff = switch2.state
                                            break;
                                        case 5:
                                            switchNo = switch3.number;
                                            switch3.state = (switch3.state == SwitchState.SwitchOff) ? SwitchState.SwitchOn : SwitchState.SwitchOff;
                                            switchOnOff = switch3.state
                                            break;
                                        case 7:
                                            switchNo = switch4.number;
                                            switch4.state = (switch4.state == SwitchState.SwitchOff) ? SwitchState.SwitchOn : SwitchState.SwitchOff;
                                            switchOnOff = switch4.state
                                            break;
                                        case 9:
                                            switchNo = switch5.number;
                                            switch5.state = (switch5.state == SwitchState.SwitchOff) ? SwitchState.SwitchOn : SwitchState.SwitchOff;
                                            switchOnOff = switch5.state
                                            break;
                                        case 11:
                                            switchNo = switch6.number;
                                            switch6.state = (switch6.state === SwitchState.SwitchOff) ? SwitchState.SwitchOn : SwitchState.SwitchOff;
                                            switchOnOff = switch6.state
                                            break;                                       
                                        default:
                                            app.debug(`Invalid Switch found ${switchNum}`);
                                            break;
                                    }

                                    if (switchOnOff == SwitchState.SwitchOn){
                                        app.debug(`SwitchOnOff State: ${switchOnOff}}`);
                                        send502(options.SwitchBank, switchNo, SwitchState.SwitchOn, path)
                                    } 
                                    else if (switchOnOff == SwitchState.SwitchOff){
                                        app.debug(`SwitchOnOff State: ${switchOnOff}}`);
                                        send502(options.SwitchBank, switchNo, SwitchState.SwitchOff, path)
                                    }
                                }
                            }
                        }
                    });                   
                }
            });        
        })     
    };

    function send502(swtchbnk, switchNum, swtchval, path){
        //Why do we need to stringify this!!!? Doesn't work when used directly.
        const instance = JSON.stringify(swtchbnk);
        const pgn = (0, ts_pgns.convertCamelCase)(app, new ts_pgns.PGN_127502({instance}));
        const sval = swtchval === 1 ? verboseOnOff.On : verboseOnOff.Off;
        pgn.fields[`switch${switchNum}`] = sval;
        app.debug(`sending ${pgn}`);
        app.emit('nmea2000JsonOut', pgn);
        app.debug(pgn);

        app.debug('***************************************************************************');
        // For debug purposes only. In practical use it will be sent by switching device.
        // So comment out when deployed.
        send501(swtchbnk, switchNum, swtchval, path);
    }

    function send501(swtchbnk, switchNum, swtchval, path){
        //Why do we need to stringify this!!!? Doesn't work when used directly.
        const instance = JSON.stringify(swtchbnk);
        const pgn = (0, ts_pgns.convertCamelCase)(app, new ts_pgns.PGN_127501({instance}));
        const sval = swtchval === 1 ? verboseOnOff.On : verboseOnOff.Off;
        pgn.fields[`indicator${switchNum}`] = sval;
        app.debug(`sending ${pgn}`);
        app.emit('nmea2000JsonOut', pgn);
        app.debug(pgn);
    }

    plugin.stop = () => {
    unsubscribes.forEach((f) => f())
    unsubscribes = []
    };

    function makeBinaryStatusReport(bank) {
        var _a;
        var pgn = new ts_pgns.PGN_127501({
            instance: bank.instance
        });
        (_a = bank.switches) === null || _a === void 0 ? void 0 : _a.forEach(function (sw, index) {
            var value = app.getSelfPath(sw);
            if (value && typeof value.value !== 'undefined') {
                ;
                pgn.fields["indicator".concat(index + 1)] =
                    value.value === 1 || value.value === true ? 'On' : 'Off';
            }
        });
        return pgn;
    }

    return plugin;
};
