package be.ucll.unit.service;

import be.ucll.model.User;
import be.ucll.repository.LoanRepository;
import be.ucll.repository.UserRepository;
import be.ucll.service.UserService;
import jakarta.validation.ConstraintViolation;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import jakarta.validation.Validation;
import jakarta.validation.ValidatorFactory;
import jakarta.validation.Validator;
import java.util.Set;

import java.util.List;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

public class UserServiceTest {

    private static ValidatorFactory validatorFactory;
    private static Validator validator;

    @BeforeAll
    public static void createValidator() {
        validatorFactory = Validation.
                buildDefaultValidatorFactory();
        validator =
                validatorFactory.getValidator();
    }

    private UserService userService;
    private UserRepository userRepository;
    private LoanRepository loanRepository;


    @BeforeEach
    public void setUp() {
        userService = new UserService(userRepository, loanRepository); // Pass both arguments
    }

    @Test
    public void testGetAllUsers() {
        List<User> users = userService.getAllUsers();
        assertEquals(5, users.size());
    }

    @Test
    public void testGetAllAdultUsers() {
        List<User> adults = userService.getAllAdultUsers();
        assertEquals(3, adults.size());
    }

    @Test
    public void testGetUsersWithinAgeRange() {
        List<User> users = userService.getUsersWithinAgeRange(5, 30);
        assertEquals(4, users.size());
    }

    @Test
    public void testGetUsersWithinAgeRangeInvalidMinMax() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            userService.getUsersWithinAgeRange(30, 5);
        });
        assertEquals("Minimum age cannot be greater than maximum age.", exception.getMessage());
    }

    @Test
    public void testGetUsersWithinAgeRangeInvalidRange() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            userService.getUsersWithinAgeRange(-1, 151);
        });
        assertEquals("Invalid age range. Age must be between 0 and 150.", exception.getMessage());
    }

    @Test
    public void testGetUsersByName() {
        List<User> users = userService.getUserByName("doe");
        assertEquals(4, users.size());
    }

    @Test
    public void testGetUsersByNameWhereFilterIsInUpperCase() {
        List<User> users = userService.getUserByName("DOE");
        assertEquals(4, users.size());
    }

    @Test
    public void testGetUsersByNameWhereFilterInputIsNotFound() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            userService.getUserByName("Doep");
        });
        assertEquals("No users found with the specified name.", exception.getMessage());
    }

    @Test
    public void testGetUsersByNameWhereFilterInputIsEmpty() {
        List<User> users = userService.getUserByName("");
        assertEquals(5, users.size());
    }

    @Test
    public void testAddUser() {
        User user = new User("Jordy Doe", 12, "jordy.doe@ucll.be", "jordy1234");
        User usr = userService.addUser(user);
        assertEquals(6, userService.getAllUsers().size());
    }

    @Test
    public void testAddUserWithInvalidAge() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            User user = new User("Jordy Doe", 111, "jordy.doe@ucll.be", "jordy1234");
//            userService.addUser(user);
//        });
//        assertEquals("Age must be a positive integer between 0 and 101.", exception.getMessage());

        User user = new User("John Doe", 102, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Age can not be greater than 101.", violations.iterator().next().getMessage());
    }

    @Test
    public void testAddUserWithInvalidEmail() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            User user = new User("Jordy Doe", 12, "jordy.doeucll.be", "jordy1234");
//            userService.addUser(user);
//        });
//        assertEquals("E-mail must be a valid email format.", exception.getMessage());

        User user = new User("John Doe", 56, "invalid email", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("E-mail must be a valid email format.", violations.iterator().next().getMessage());
    }

    @Test
    public void testAddUserWithInvalidName() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            User user = new User("", 12, "jordy.doeucll.be", "jordy1234");
//            userService.addUser(user);
//        });
//        assertEquals("Name is required.", exception.getMessage());

        User user = new User("", 56, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Name is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void testAddUserWithInvalidPassword() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            User user = new User("Jordy Doe", 12, "jordy.doe@ucll.be", "jordy");
//            userService.addUser(user);
//        });
//        assertEquals("Password must be at least 8 characters long.", exception.getMessage());

        User user = new User("John Doe", 56, "john.doe@ucll.be", "short");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Password must be at least 8 characters long.", violations.iterator().next().getMessage());
    }

    @Test
    public void testAddUserWithEmailThatAlreadyExists() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            User user = new User("John Doe", 12, "john.doe@ucll.be", "john1234");
            userService.addUser(user);
        });
        assertEquals("User already exists.", exception.getMessage());
    }

    @Test
    public void testUpdateUserName() {
        User newInformation = new User("Johnny Doe", 25, "john.doe@ucll.be", "john1234");
        User user = userService.updateUser("john.doe@ucll.be", newInformation);

        assertEquals("Johnny Doe", user.getName());
    }

    @Test
    public void testUpdateUserAge() {
        User newInformation = new User("John Doe", 26, "john.doe@ucll.be", "john1234");
        User user = userService.updateUser("john.doe@ucll.be", newInformation);

        assertEquals(26, user.getAge());
    }

    @Test
    public void testUpdateUserPassword() {
        User newInformation = new User("John Doe", 25, "john.doe@ucll.be", "john12345");
        User user = userService.updateUser("john.doe@ucll.be", newInformation);

        assertEquals("john12345", user.getPassword());
    }

    @Test
    public void testUpdateUserWhenEmailIsNotFound() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            User newInformation = new User("John Doe", 25, "john.doe@ucll.be", "john1234");
            User user = userService.updateUser("nonexistent@ucll.be", newInformation);
        });
        assertEquals("User does not exist.", exception.getMessage());
    }

    @Test
    public void testUpdateUserWhenAgeIsOutOfRange() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            User newInformation = new User("John Doe", 111, "john.doe@ucll.be", "john1234");
//            User user = userService.updateUser("john.doe@ucll.be", newInformation);
//        });
//        assertEquals("Age must be a positive integer between 0 and 101.", exception.getMessage());

        User user = new User("John Doe", 102, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Age can not be greater than 101.", violations.iterator().next().getMessage());
    }

    @Test
    public void testUpdateUserWhenTryingToUpdateEmail() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            User newInformation = new User("John Doe", 25, "johnny.doe@ucll.be", "john1234");
            User user = userService.updateUser("john.doe@ucll.be", newInformation);
        });
        assertEquals("Email cannot be changed.", exception.getMessage());
    }
    @Test
    public void testDeleteUserSuccessfully() {
        String result = userService.deleteUser("john.doe@ucll.be");
        assertEquals("User successfully deleted.", result);
    }

    @Test
    public void testDeleteUserWithActiveLoans() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            userService.deleteUser("jane.toe@ucll.be");
        });
        assertEquals("User has active loans.", exception.getMessage());
    }

    @Test
    public void testDeleteNonExistentUser() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            userService.deleteUser("nonexistent@ucll.be");
        });
        assertEquals("User does not exist.", exception.getMessage());
    }

    @AfterAll
    public static void close() {
        validatorFactory.close();
    }
}