package be.ucll.unit.model;

import be.ucll.model.User;
import jakarta.validation.ConstraintViolation;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import jakarta.validation.ConstraintViolation;
import jakarta.validation.Validation;
import jakarta.validation.ValidatorFactory;
import jakarta.validation.Validator;
import java.util.Set;

import static org.junit.jupiter.api.Assertions.*;

public class UserTest {

    private static ValidatorFactory validatorFactory;
    private static Validator validator;

    @BeforeAll
    public static void createValidator() {
        validatorFactory = Validation.
                buildDefaultValidatorFactory();
        validator =
                validatorFactory.getValidator();
    }

    @Test
    public void givenValidValues_whenCreatingUser_thenUserIsCreatedWithThoseValues() {
        User user = new User("John Doe", 56, "john.doe@ucll.be", "john1234");
        assertEquals("John Doe", user.getName());
        assertEquals(56, user.getAge());
        assertEquals("john.doe@ucll.be", user.getEmail());
        assertEquals("john1234", user.getPassword());
    }

    @Test
    public void givenNegativeAge_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("John Doe", -1, "john.doe@ucll.be", "john1234");
//        });
//        assertEquals("Age must be a positive integer between 0 and 101.", exception.getMessage());

        User user = new User("John Doe", -1, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Age must be a positive integer.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenAgeGreaterThan101_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("John Doe", 102, "john.doe@ucll.be", "john1234");
//        });
//        assertEquals("Age must be a positive integer between 0 and 101.", exception.getMessage());

        User user = new User("John Doe", 102, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Age can not be greater than 101.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenInvalidName_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("", 56, "john.doe@ucll.be", "john1234");
//        });
//        assertEquals("Name is required", exception.getMessage());

        User user = new User("", 56, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Name is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenNameWithOnlySpaces_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("   ", 56, "john.doe@ucll.be", "john1234");
//        });
//        assertEquals("Name is required", exception.getMessage());

        User user = new User("    ", 56, "john.doe@ucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Name is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenInvalidEmail_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("John Doe", 56, "invalid-email", "john1234");
//        });
//        assertEquals("E-mail must be a valid email format.", exception.getMessage());

        User user = new User("John Doe", 56, "invalid email", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("E-mail must be a valid email format.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenEmailWithoutAtSymbol_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("John Doe", 56, "john.doeucll.be", "john1234");
//        });
//        assertEquals("E-mail must be a valid email format.", exception.getMessage());

        User user = new User("John Doe", 56, "john.doeucll.be", "john1234");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("E-mail must be a valid email format.", violations.iterator().next().getMessage());
    }


    // doesnt work because email checker olny checks for @ and not for dot
    @Test
    public void givenEmailWithoutDot_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("John Doe", 56, "john@doeucllbe", "john1234");
//        });
//        assertEquals("E-mail must be a valid email format.", exception.getMessage());

//        User user = new User("John Doe", 56, "johndoe@ucllbe", "john1234");
//        Set<ConstraintViolation<User>> violations = validator.validate(user);
//        assertEquals(1, violations.size());
//        assertEquals("E-mail must be a valid email format.", violations.iterator().next().getMessage());

    }

    @Test
    public void givenInvalidPassword_whenCreatingUser_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new User("John Doe", 56, "john.doe@ucll.be", "short");
//        });
//        assertEquals("Password must be at least 8 characters long", exception.getMessage());

        User user = new User("John Doe", 56, "john.doe@ucll.be", "short");
        Set<ConstraintViolation<User>> violations = validator.validate(user);
        assertEquals(1, violations.size());
        assertEquals("Password must be at least 8 characters long.", violations.iterator().next().getMessage());
    }

    @AfterAll
    public static void close() {
        validatorFactory.close();
    }
}