package be.ucll.unit.model;

import be.ucll.model.Book;
import be.ucll.model.Magazine;
import jakarta.validation.ConstraintViolation;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.*;

import jakarta.validation.Validation;
import jakarta.validation.ValidatorFactory;
import jakarta.validation.Validator;
import java.util.Set;

public class MagazineTest {

    private static ValidatorFactory validatorFactory;
    private static Validator validator;

    @BeforeAll
    public static void createValidator() {
        validatorFactory = Validation.
                buildDefaultValidatorFactory();
        validator =
                validatorFactory.getValidator();
    }

    @Test
    public void givenValidValues_whenCreatingMagazine_thenMagazineIsCreatedWithThoseValues() {
        Magazine magazine = new Magazine("Effective Java", "Joshua Bloch", "12345678", 2008, 10);
        assertEquals("Effective Java", magazine.getTitle());
        assertEquals("Joshua Bloch", magazine.getEditor());
        assertEquals("12345678", magazine.getISSN());
        assertEquals(2008, magazine.getPublicationYear());
        assertEquals(10, magazine.getAvailableCopies());
    }

    @Test
    public void givenInvalidTitle_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("", "Joshua Bloch", "12345678", 2008, 10);
//        });
//        assertEquals("Title is required", exception.getMessage());

        Magazine magazine = new Magazine("", "Joshua Bloch", "12345678", 2008, 10);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("Title is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenTitleWithOnlySpaces_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("   ", "Joshua Bloch", "12345678", 2008, 10);
//        });
//        assertEquals("Title is required", exception.getMessage());

        Magazine magazine = new Magazine("   ", "Joshua Bloch", "12345678", 2008, 10);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("Title is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenInvalidEditor_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("Effective Java", "", "12345678", 2008, 10);
//        });
//        assertEquals("Editor is required", exception.getMessage());

        Magazine magazine = new Magazine("Effective Java", "", "12345678", 2008, 10);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("Editor is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenEditorWithOnlySpaces_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("Effective Java", "   ", "12345678", 2008, 10);
//        });
//        assertEquals("Editor is required", exception.getMessage());

        Magazine magazine = new Magazine("Effective Java", "    ", "12345678", 2008, 10);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("Editor is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenEmptyISSN_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("Effective Java", "Joshua Bloch", "", 2008, 10);
//        });
//        assertEquals("ISSN is required", exception.getMessage());

        Magazine magazine = new Magazine("Effective Java", "Joshua Bloch", "", 2008, 10);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("ISSN is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenISSNWithOnlySpaces_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("Effective Java", "Joshua Bloch", "   ", 2008, 10);
//        });
//        assertEquals("ISSN is required", exception.getMessage());

        Magazine magazine = new Magazine("Effective Java", "Joshua Bloch", "    ", 2008, 10);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("ISSN is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenInvalidPublicationYear_whenCreatingMagazine_thenThrowsException() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Magazine("Effective Java", "Joshua Bloch", "12345678", -1, 10);
        });
        assertEquals("Publication year must be a positive integer", exception.getMessage());
    }

    @Test
    public void givenFuturePublicationYear_whenCreatingMagazine_thenThrowsException() {
        int nextYear = java.time.Year.now().getValue() + 1;
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Magazine("Effective Java", "Joshua Bloch", "12345678", nextYear, 10);
        });
        assertEquals("Publication year cannot be in the future", exception.getMessage());
    }

    @Test
    public void givenInvalidAvailableCopies_whenCreatingMagazine_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Magazine("Effective Java", "Joshua Bloch", "12345678", 2008, -1);
//        });
//        assertEquals("Available copies must be a positive number", exception.getMessage());

        Magazine magazine = new Magazine("Effective Java", "Joshua Bloch", "12345678", 2008, -1);
        Set<ConstraintViolation<Magazine>> violations = validator.validate(magazine);
        assertEquals(1, violations.size());
        assertEquals("Available copies must be a positive number.", violations.iterator().next().getMessage());
    }

    @AfterAll
    public static void close() {
        validatorFactory.close();
    }
}