package be.ucll.unit.model;

import be.ucll.model.Book;
import be.ucll.model.Loan;
import be.ucll.model.Publication;
import jakarta.validation.ConstraintViolation;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import java.time.LocalDate;
import static org.junit.jupiter.api.Assertions.*;

import jakarta.validation.Validation;
import jakarta.validation.ValidatorFactory;
import jakarta.validation.Validator;
import java.util.Set;

public class LoanTest {

    private static ValidatorFactory validatorFactory;
    private static Validator validator;

    @BeforeAll
    public static void createValidator() {
        validatorFactory = Validation.
                buildDefaultValidatorFactory();
        validator =
                validatorFactory.getValidator();
    }

    @Test
    public void givenValidValues_whenCreatingLoan_thenLoanIsCreatedWithThoseValues() {
        String userEmail = "john.doe@example.com";
        String publicationTitle = "Effective Java";
        LocalDate startDate = LocalDate.now();
        LocalDate endDate = startDate.plusDays(21);
        Loan loan = new Loan(userEmail, publicationTitle, startDate, endDate);
        assertEquals(userEmail, loan.getUserEmail());
        assertEquals(publicationTitle, loan.getPublicationTitle());
        assertEquals(startDate, loan.getStartDate());
        assertEquals(endDate, loan.getEndDate());
    }

    @Test
    public void givenNullUserEmail_whenCreatingLoan_thenThrowsException() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Loan(null, "Effective Java", LocalDate.now(), LocalDate.now().plusDays(21));
        });
        assertEquals("User email is required.", exception.getMessage());

//        Loan loan = new Loan(null, "Effective Java", LocalDate.now(), LocalDate.now().plusDays(21));
//        Set<ConstraintViolation<Loan>> violations = validator.validate(loan);
//        assertEquals(1, violations.size());
//        assertEquals("User email is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenNullPublicationTitle_whenCreatingLoan_thenThrowsException() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Loan("john.doe@example.com", null, LocalDate.now(), LocalDate.now().plusDays(21));
        });
        assertEquals("List is required.", exception.getMessage());
    }

    @Test
    public void givenFutureStartDate_whenCreatingLoan_thenThrowsException() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Loan("john.doe@example.com", "Effective Java", LocalDate.now().plusDays(1), LocalDate.now().plusDays(22));
        });
        assertEquals("Start date cannot be in the future.", exception.getMessage());
    }

    @Test
    public void givenValidValues_whenReturningLoan_thenAvailableCopiesAreIncremented() {
        String userEmail = "john.doe@example.com";
        String publicationTitle = "Effective Java";
        LocalDate startDate = LocalDate.now();
        LocalDate endDate = startDate.plusDays(21);
        Publication publication = new Publication(publicationTitle, 2021, 5);
        Loan loan = new Loan(userEmail, publicationTitle, startDate, endDate);
        publication.returnPublication();
        assertEquals(6, publication.getAvailableCopies());
    }

    @Test
    public void givenLoanWithStartDateInPast_whenCreatingLoan_thenLoanIsCreatedWithThoseValues() {
        String userEmail = "john.doe@example.com";
        String publicationTitle = "Effective Java";
        LocalDate pastDate = LocalDate.now().minusDays(1);
        Loan loan = new Loan(userEmail, publicationTitle, pastDate, pastDate.plusDays(21));
        assertEquals(userEmail, loan.getUserEmail());
        assertEquals(publicationTitle, loan.getPublicationTitle());
        assertEquals(pastDate, loan.getStartDate());
        assertEquals(pastDate.plusDays(21), loan.getEndDate());
    }

    @Test
    public void givenLoanWithStartDateToday_whenCreatingLoan_thenLoanIsCreatedWithThoseValues() {
        String userEmail = "john.doe@example.com";
        String publicationTitle = "Effective Java";
        LocalDate today = LocalDate.now();
        Loan loan = new Loan(userEmail, publicationTitle, today, today.plusDays(21));
        assertEquals(userEmail, loan.getUserEmail());
        assertEquals(publicationTitle, loan.getPublicationTitle());
        assertEquals(today, loan.getStartDate());
        assertEquals(today.plusDays(21), loan.getEndDate());
    }

    @Test
    public void givenLoanWithStartDateInFuture_whenCreatingLoan_thenThrowsException() {
        String userEmail = "john.doe@example.com";
        String publicationTitle = "Effective Java";
        LocalDate futureDate = LocalDate.now().plusDays(1);
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Loan(userEmail, publicationTitle, futureDate, futureDate.plusDays(21));
        });
        assertEquals("Start date cannot be in the future.", exception.getMessage());
    }

    @AfterAll
    public static void close() {
        validatorFactory.close();
    }
}