package be.ucll.unit.model;

import be.ucll.model.Book;
import be.ucll.model.User;
import jakarta.validation.ConstraintViolation;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import jakarta.validation.ConstraintViolation;
import jakarta.validation.Validation;
import jakarta.validation.ValidatorFactory;
import jakarta.validation.Validator;
import java.util.Set;
import java.util.stream.Collectors;

import static org.junit.jupiter.api.Assertions.*;

public class BookTest {

    private static ValidatorFactory validatorFactory;
    private static Validator validator;

    @BeforeAll
    public static void createValidator() {
        validatorFactory = Validation.
                buildDefaultValidatorFactory();
        validator =
                validatorFactory.getValidator();
    }

    @Test
    public void givenValidValues_whenCreatingBook_thenBookIsCreatedWithThoseValues() {
        Book book = new Book("Effective Java", "Joshua Bloch", "1234567890123", 2008, 5);
        assertEquals("Effective Java", book.getTitle());
        assertEquals("Joshua Bloch", book.getAuthor());
        assertEquals("1234567890123", book.getISBN());
        assertEquals(2008, book.getPublicationYear());
        assertEquals(5, book.getAvailableCopies());
    }

    @Test
    public void givenInvalidTitle_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("", "Joshua Bloch", "1234567890123", 2008, 5);
//        });
//        assertEquals("Title is required", exception.getMessage());

        Book book = new Book("", "Joshua Bloch", "1234567890123", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(1, violations.size());
        assertEquals("Title is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenTitleWithOnlySpaces_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("   ", "Joshua Bloch", "1234567890123", 2008, 5);
//        });
//        assertEquals("Title is required", exception.getMessage());

        Book book = new Book("    ", "Joshua Bloch", "1234567890123", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(1, violations.size());
        assertEquals("Title is required.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenInvalidAuthor_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("Effective Java", "", "1234567890123", 2008, 5);
//        });
//        assertEquals("Author is required", exception.getMessage());


        Book book = new Book("Effective Java", "", "1234567890123", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(1, violations.size());
        assertEquals("Author is required", violations.iterator().next().getMessage());
    }

    @Test
    public void givenAuthorWithOnlySpaces_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("Effective Java", "   ", "1234567890123", 2008, 5);
//        });
//        assertEquals("Author is required", exception.getMessage());

        Book book = new Book("Effective Java", "    ", "1234567890123", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(1, violations.size());
        assertEquals("Author is required", violations.iterator().next().getMessage());
    }

    @Test
    public void givenEmptyISBN_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("Effective Java", "Joshua Bloch", "", 2008, 5);
//        });
//        assertEquals("ISBN is required", exception.getMessage());

        Book book = new Book("Effective Java", "Joshua Bloch", "", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(2, violations.size());
        Set<String> messages = violations.stream()
                .map(ConstraintViolation::getMessage)
                .collect(Collectors.toSet());

        assertTrue(messages.contains("ISBN is required."));
        assertTrue(messages.contains("ISBN must contain exactly 13 digits."));

    }

    @Test
    public void givenISBNWithOnlySpaces_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("Effective Java", "Joshua Bloch", "   ", 2008, 5);
//        });
//        assertEquals("ISBN is required", exception.getMessage());

        Book book = new Book("Effective Java", "Joshua Bloch", "    ", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(2, violations.size());
        Set<String> messages = violations.stream()
                .map(ConstraintViolation::getMessage)
                .collect(Collectors.toSet());

        assertTrue(messages.contains("ISBN is required."));
        assertTrue(messages.contains("ISBN must contain exactly 13 digits."));
    }

    @Test
    public void givenInvalidISBN_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("Effective Java", "Joshua Bloch", "invalid-isbn", 2008, 5);
//        });
//        assertEquals("ISBN must contain exactly 13 digits", exception.getMessage());

        Book book = new Book("Effective Java", "Joshua Bloch", "invalid-isbn", 2008, 5);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(1, violations.size());
        assertEquals("ISBN must contain exactly 13 digits.", violations.iterator().next().getMessage());
    }

    @Test
    public void givenInvalidPublicationYear_whenCreatingBook_thenThrowsException() {
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Book("Effective Java", "Joshua Bloch", "1234567890123", -1, 5);
        });
        assertEquals("Publication year must be a positive integer", exception.getMessage());
    }

    @Test
    public void givenFuturePublicationYear_whenCreatingBook_thenThrowsException() {
        int nextYear = java.time.Year.now().getValue() + 1;
        Exception exception = assertThrows(RuntimeException.class, () -> {
            new Book("Effective Java", "Joshua Bloch", "1234567890123", nextYear, 5);
        });
        assertEquals("Publication year cannot be in the future", exception.getMessage());
    }

    @Test
    public void givenInvalidAvailableCopies_whenCreatingBook_thenThrowsException() {
//        Exception exception = assertThrows(RuntimeException.class, () -> {
//            new Book("Effective Java", "Joshua Bloch", "1234567890123", 2008, -1);
//        });
//        assertEquals("Available copies must be a positive number.", exception.getMessage());

        Book book = new Book("Effective Java", "Joshua Bloch", "1234567890123", 2008, -1);
        Set<ConstraintViolation<Book>> violations = validator.validate(book);
        assertEquals(1, violations.size());
        assertEquals("Available copies must be a positive number.", violations.iterator().next().getMessage());
    }

    @AfterAll
    public static void close() {
        validatorFactory.close();
    }
}