package be.ucll.integration;

import be.ucll.repository.DbInitializer;
import be.ucll.repository.UserRepository;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.reactive.AutoConfigureWebTestClient;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.jdbc.Sql;
import org.springframework.test.web.reactive.server.WebTestClient;

import static org.junit.jupiter.api.Assertions.assertEquals;

@SpringBootTest
@AutoConfigureWebTestClient
@Sql("classpath:schema.sql")
public class UserIntegrationTest {

    @Autowired
    private DbInitializer dbInitializer;

    @Autowired
    private UserRepository userRepository;

    @Autowired
    private WebTestClient webTestClient;

    @BeforeEach
    public void setup() {
        dbInitializer.initialize();
    }

    @Test
    public void givenUsers_whenGetUsers_thenUsersAreReturned() {
        webTestClient
                .get()
                .uri("/users")
                .exchange()
                .expectStatus().isOk()
                .expectBody()
                .jsonPath("$.length()").isEqualTo(1);
    }

    @Test
    public void givenUsers_whenGetAdultUsers_thenAdultsAreReturned() {
        webTestClient
                .get()
                .uri("/users/adults")
                .exchange()
                .expectStatus().isOk()
                .expectBody()
                .jsonPath("$.length()").isEqualTo(1);
    }

    @Test
    public void givenUsers_whenGetUsersWithinAgeRange_thenUsersAreReturned() {
        webTestClient
                .get()
                .uri(uriBuilder -> uriBuilder.path("/users/age/{min}/{max}")
                        .build(20, 30))
                .exchange()
                .expectStatus().isOk()
                .expectBody()
                .jsonPath("$.length()").isEqualTo(1);
    }

    @Test
    public void givenUser_whenGetUserLoans_thenLoansAreReturned() {
        webTestClient
                .get()
                .uri("/users/john.doe@ucll.be/loans")
                .exchange()
                .expectStatus().isOk()
                .expectBody()
                .jsonPath("$.length()").isEqualTo(1);
    }

    @Test
    public void givenNewUser_whenAddUser_thenUserIsAdded() {
        webTestClient
                .post()
                .uri("/users")
                .header("Content-Type", "application/json")
                .bodyValue("{\"name\":\"Jane Doe\",\"age\":30,\"email\":\"jane.doe@ucll.be\",\"password\":\"jane1234\"}")
                .exchange()
                .expectStatus().isOk()
                .expectBody()
                .jsonPath("$.name").isEqualTo("Jane Doe")
                .jsonPath("$.age").isEqualTo(30)
                .jsonPath("$.email").isEqualTo("jane.doe@ucll.be");
    }

    @Test
    public void givenUpdatedUser_whenUpdateUser_thenUserIsUpdated() {
        webTestClient
                .put()
                .uri("/users/john.doe@ucll.be")
                .header("Content-Type", "application/json")
                .bodyValue("{\"name\":\"Johnny Doe\",\"age\":26,\"email\":\"john.doe@ucll.be\",\"password\":\"john12345\"}")
                .exchange()
                .expectStatus().isOk()
                .expectBody()
                .jsonPath("$.name").isEqualTo("Johnny Doe")
                .jsonPath("$.age").isEqualTo(26)
                .jsonPath("$.email").isEqualTo("john.doe@ucll.be");
    }

    @Test
    public void givenUser_whenDeleteUser_thenUserIsDeleted() {
        webTestClient
                .delete()
                .uri("/users/john.doe@ucll.be")
                .exchange()
                .expectStatus().isOk()
                .expectBody(String.class)
                .isEqualTo("User successfully deleted.");
        assertEquals(0, userRepository.findAll().size());
    }
}