package be.ucll.controller;


import be.ucll.model.Loan;
import be.ucll.model.User;
import be.ucll.repository.UserRepository;
import be.ucll.service.LoanService;
import be.ucll.service.UserService;
import jakarta.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;


@RestController
@RequestMapping("/users")
public class UserRestController {

    private final UserService userService;
    private final LoanService loanService;
    private final UserRepository userRepository;

    @Autowired
    public UserRestController(UserService userService, LoanService loanService, UserRepository userRepository) {
        this.userService = userService;
        this.loanService = loanService;
        this.userRepository = userRepository;
    }
//    @DeleteMapping("/reset")
//    public ResponseEntity<Void> resetUsers() {
//        userRepository.resetRepositoryData();
//        return ResponseEntity.ok().build();
//    }

    @GetMapping
    public List<User> getAllUsers(@RequestParam(value = "name", required = false) String name) {
        if (name == null) {
            return userService.getAllUsers();
        } else {
            return userService.getUserByName(name);
        }
    }

    @GetMapping("/adults")
    public List<User> getAllAdultUsers() {
        return userService.getAllAdultUsers();
    }

    @GetMapping("/age/{min}/{max}")
    public List<User> getUsersWithinAgeRange(@PathVariable int min, @PathVariable int max) {
        return userService.getUsersWithinAgeRange(min, max);
    }

    @GetMapping("/{email}/loans")
    public List<Loan> getUserLoans(@PathVariable String email,
                                   @RequestParam(value = "onlyActive", required = false, defaultValue = "false") boolean onlyActive) {
        return loanService.getLoansByUser(email, onlyActive);
    }

    @PostMapping
    public User addUser(@Valid @RequestBody User user) {
        return userService.addUser(user);
    }

    @PutMapping("/{email}")
    public User updateUser(@PathVariable String email, @Valid @RequestBody User user) {
        return userService.updateUser(email, user);
    }

    @DeleteMapping("/{email}/loans")
    public String removeUserLoans(@PathVariable String email) {
        return loanService.removeUserLoans(email);
    }
    @DeleteMapping("/{email}")
    public String deleteUser(@PathVariable String email) {
        return userService.deleteUser(email);
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ExceptionHandler({RuntimeException.class})
    public Map<String, String> handleRuntimeException(RuntimeException ex) {
        Map<String, String> errors = new HashMap<>();
        errors.put("error", ex.getMessage());
        return errors;
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ExceptionHandler({MethodArgumentNotValidException.class})
    public Map<String, String> handleValidationExceptions(MethodArgumentNotValidException ex) {
        Map<String, String> errors = new HashMap<>();
        for (FieldError error : ex.getFieldErrors()) {
            String fieldName = error.getField();
            String errorMessage = error.getDefaultMessage();
            errors.put(fieldName, errorMessage);
        }
        return errors;
    }
}